# Copyright (C) 2023 Intel Corporation
# SPDX-License-Identifier: MIT

import re
from pathlib import Path

from mpp.parsers.system_information_parser import SystemInformationParser, \
    SystemInfoParserAttributes, SectionNames, UnitNames, SymbolTable, SymbolNames
from mpp.core.api_args import SystemInformation

unit_names = UnitNames('OS Processor', 'Phys. Package', 'Module', 'Core Type', 'Core', 'Logical Processor')

section_names = SectionNames('Processor Features', 'OS Processor <', 'System Features', 'Uncore Performance Monitoring '
                             'Units', 'EDP Configuration', 'RDT H/W Support', 'GPU Information', 'RAM Features',
                             'QPI Link Features', 'IIO Unit Features', 'PCIeX')

symbol_names = SymbolNames('Number of Packages', 'Threads Per Core', 'Cores Per Package', 'Threads Per Package')


class EmonSystemInformationParser(SystemInformationParser):
    INT_LIKE_RE = re.compile(r'\d+[\d,]*')

    """
    Parse system information stored in EMON data files (emon.dat, emon-v.dat)
    """

    parser_attributes = SystemInfoParserAttributes("Version Info:", '\t', unit_names, section_names, symbol_names)

    def __init__(self, input_file: Path, ref_tsc_hz: int = 0):
        """
        Initialize the EMON system information parser

        :param input_file: the EMON data file to parse
        :param ref_tsc_hz: an optional system frequency value (in Hz). Overrides system information in the input file
                            (if such information exists)

        """

        super().__init__(input_file, ref_tsc_hz)
        self._set_qpi_link_speed()

    def _set_ref_tsc(self):
        self._ref_tsc = self.extract_attribute_value('TSC Freq', 'Processor Base Freq', 1000000)
    
    def _set_qpi_link_speed(self):
        self.qpi_link_speed = self.extract_attribute_value('QPI Link Speed', None, 1000000000)

    def extract_attribute_value(self, attribute, secondary_attribute, exponent_factor):
        candidates = [key for key in self.attributes.keys() if key.startswith(attribute)]
        if len(candidates) > 0:
            # Assume the first occurrence inside candidates is the desired attribute
            attribute_value = self.attributes[candidates[0]].split(' ', 1)[0]
        elif secondary_attribute and secondary_attribute in self.attributes:
            attribute_value = self.attributes[secondary_attribute].split(' ', 1)[0]
        else:
            # Unable to determine which attribute
            attribute_value = self.set_null_attribute_val(attribute)
        if attribute_value:
            attribute_value = float(attribute_value) * exponent_factor
        return attribute_value
    
    @staticmethod
    def set_null_attribute_val(attribute):
        if 'TSC Freq' in attribute:
            return 0
        return None
    