#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to traverse the topdown tree and collect data from
# requested fields for pretty-printing.
# ------------------------------------------------------------------------------

import sys

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 2:
    print('Usage: "python {} path_to_project_dir [comma-separated-column-list]"'.format(__file__))
    sys.exit(2)

# Open the Advisor Project and load the data.
project = advisor.open_project(sys.argv[1])
data = project.load(advisor.SURVEY)

name_key = "function_call_sites_and_loops"
keys = []
if len(sys.argv) > 2:
    keys += sys.argv[2].split(",")

lenghts = {key: len(key) for key in keys}
lenghts[name_key] = len(name_key)

# Traverse the topdown tree and determine the max lenghts of requested fields.
# For 'name_key' we also take into account depth to show it with indentation.
for row in data.topdown:
    stack = [(row, 0)]
    while stack:
        v, level = stack.pop()
        for r in v.get_children():
            stack.append((r, level + 1))
        for key in keys:
            lenghts[key] = max(lenghts[key], len(v[key]))
        lenghts[name_key] = max(lenghts[name_key], len(v[name_key]) + level * 2)

# Prepare formats with enough space to show all fields.
fmt = "{:<" + str(max(lenghts[name_key], 32)) + "} "
for key in keys:
    fmt += "{:<" + str(max(lenghts[key], 8)) + "} "

# Traverse topdown tree again and pretty-print requested fields.
print(fmt.format(*([name_key] + keys)))
for row in data.topdown:
    stack = [(row, 0)]
    while stack:
        v, level = stack.pop()
        for r in v.get_children():
            stack.append((r, level + 1))
        print(fmt.format(*([" " * level * 2 + v[name_key]] + [v[key] for key in keys])))
