/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include "lwpmudrv_defines.h"

#include "lwpmudrv_types.h"
#include "rise_errors.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv.h"
#include "control.h"
#include "utility.h"

#include <sys/smp.h>
#include <sys/cpuset.h>

#ifndef mb
#define mb()	__asm volatile("mfence" ::: "memory")
#endif

/*
 *  Global State Nodes - keep here for now.  Abstract out when necessary.
 */
GLOBAL_STATE_NODE  driver_state;
MSR_DATA           msr_data      = NULL;

/* ------------------------------------------------------------------------- */
/*!
 * @fn       VOID CONTROL_Invoke_Cpu (func, ctx, arg)
 *
 * @brief    Set up a DPC call and insert it into the queue
 *
 * @param    IN cpu_idx  - the core id to dispatch this function to
 *           IN func     - function to be invoked by the specified core(s)
 *           IN ctx      - pointer to the parameter block for each function
 *                         invocation
 *
 * @return   None
 *
 * <I>Special Notes:</I>
 *
 */
extern VOID
CONTROL_Invoke_Cpu (
    int     cpu_idx,
    VOID    (*func)(PVOID),
    PVOID   ctx
)
{
    SEP_DRV_LOG_TRACE_IN("Cpu_idx: %d, func: %p, ctx: %p.", cpu_idx, func, ctx);

    if (cpu_idx < 0) {
        SEP_DRV_LOG_ERROR_TRACE_OUT("Cpu_idx < 0!");
        return;
    }

    GLOBAL_STATE_cpu_count(driver_state) = 0;
    GLOBAL_STATE_dpc_count(driver_state) = 0;

    preempt_disable();
    mb();

#if __FreeBSD_version >= 900040
    cpuset_t cs;

    CPU_ZERO(&cs);
    CPU_SET(cpu_idx, &cs);
#else
    cpumask_t cs;

    cs = 0;
    cs |= (1 << cpu_idx);
#endif
    smp_rendezvous_cpus(cs, NULL, func, NULL, ctx);

    preempt_enable();

    SEP_DRV_LOG_TRACE_OUT("");
    return;
}

/* ------------------------------------------------------------------------- */
/*
 * @fn VOID CONTROL_Invoke_Parallel_Service(func, ctx, blocking, exclude)
 *
 * @param    func     - function to be invoked by each core in the system
 * @param    ctx      - pointer to the parameter block for each function invocation
 * @param    blocking - Wait for invoked function to complete
 * @param    exclude  - exclude the current core from executing the code
 *
 * @returns  None
 *
 * @brief    Service routine to handle all kinds of parallel invoke on all CPU calls
 *
 * <I>Special Notes:</I>
 *           Invoke the function provided in parallel in either a blocking or
 *           non-blocking mode.  The current core may be excluded if desired.
 *           NOTE - Do not call this function directly from source code.
 *           Use the aliases CONTROL_Invoke_Parallel(), CONTROL_Invoke_Parallel_NB(),
 *           or CONTROL_Invoke_Parallel_XS().
 *
 */
extern VOID
CONTROL_Invoke_Parallel_Service (
    VOID   (*func)(PVOID),
    PVOID  ctx,
    int    blocking,
    int    exclude
)
{
    SEP_DRV_LOG_TRACE_IN("Func: %p, ctx: %p, blocking: %d, exclude: %d.", func, ctx, blocking, exclude);

    GLOBAL_STATE_cpu_count(driver_state) = 0;
    GLOBAL_STATE_dpc_count(driver_state) = 0;

    preempt_disable();
    mb();

    if (!exclude) {
        smp_rendezvous(NULL, func, NULL, ctx);
    } else {
#if __FreeBSD_version >= 900040
        cpuset_t cs;

        cs = all_cpus;
        CPU_CLR(curcpu, &cs);
#else
        cpumask_t cs;

        cs = all_cpus;
        cs &= ~(1 << curcpu);
#endif
        smp_rendezvous_cpus(cs, NULL, func, NULL, ctx);
    }
    preempt_enable();

    SEP_DRV_LOG_TRACE_OUT("");
    return;
}

/* ------------------------------------------------------------------------- */
/*
 * @fn PVOID CONTROL_Allocate_Memory(size)
 *
 * @param    IN size     - size of the memory to allocate
 *
 * @returns  char*       - pointer to the allocated memory block
 *
 * @brief    Allocate and zero memory
 *
 * <I>Special Notes:</I>
 *           Allocate memory in the GFP_KERNEL pool.
 *
 *           Use this if memory is to be allocated within a context where
 *           the allocator can block the allocation (e.g., by putting
 *           the caller to sleep) while it tries to free up memory to
 *           satisfy the request.  Otherwise, if the allocation must
 *           occur atomically (e.g., caller cannot sleep), then use
 *           CONTROL_Allocate_KMemory instead.
 */
extern PVOID
CONTROL_Allocate_Memory (
    size_t size
)
{
    PVOID location;

    SEP_DRV_LOG_ALLOC_IN("Will allocate %u bytes...", size);

    location = (PVOID)malloc(size, M_SEP, M_WAITOK);

    if (!location) {
        SEP_DRV_LOG_ERROR_ALLOC_OUT("Failed allocation for %u bytes!", size);
        return NULL;
    }

    memset(location, 0, size);

    SEP_DRV_LOG_ALLOC_OUT("Success: %p (%uB).", location, size);
    return location;
}

/* ------------------------------------------------------------------------- */
/*
 * @fn PVOID CONTROL_Allocate_KMemory(size)
 *
 * @param    IN size     - size of the memory to allocate
 *
 * @returns  char*       - pointer to the allocated memory block
 *
 * @brief    Allocate and zero memory
 *
 * <I>Special Notes:</I>
 *           Allocate memory in the GFP_ATOMIC pool.
 *
 *           Use this if memory is to be allocated within a context where
 *           the allocator cannot block the allocation (e.g., by putting
 *           the caller to sleep) as it tries to free up memory to
 *           satisfy the request.  Examples include interrupt handlers,
 *           process context code holding locks, etc.
 */
extern PVOID
CONTROL_Allocate_KMemory (
    size_t size
)
{
    PVOID location;

    SEP_DRV_LOG_ALLOC_IN("Will allocate %u bytes...", size);

	location = contigmalloc(size, M_SEP, M_NOWAIT, 0ul, ~0ul, PAGE_SIZE, 1024 * 1024);

    if (!location) {
        SEP_DRV_LOG_ERROR_ALLOC_OUT("Failed allocation for %u bytes!", size);
        return NULL;
    }

    memset(location, 0, size);

    SEP_DRV_LOG_ALLOC_OUT("Success: %p (%uB).", location, size);
    return location;
}

/* ------------------------------------------------------------------------- */
/*
 * @fn PVOID CONTROL_Free_Memory(location)
 *
 * @param    IN location  - size of the memory to allocate
 *
 * @returns  pointer to the allocated memory block
 *
 * @brief    Frees the memory block
 *
 * <I>Special Notes:</I>
 *           Does not try to free memory if fed with a NULL pointer
 *           Expected usage:
 *               ptr = CONTROL_Free_Memory(ptr);
 *           Does not do compaction ... can have "holes" in
 *           mem_tracker list after this operation.
 */
extern PVOID
CONTROL_Free_Memory (
    PVOID  location
)
{
    SEP_DRV_LOG_ALLOC_IN("Will free %p...", location);

    if (location) {
        free(location, M_SEP);
    }

    SEP_DRV_LOG_ALLOC_OUT("Freed %p.", location);
    return NULL;
}

/* ------------------------------------------------------------------------- */
/*
 * @fn VOID CONTROL_Free_KMemory(location, size)
 *
 * @param    IN location  - buffer to free
 *           IN size      - size of buffer to free
 *
 * @brief    Frees the memory block
 *
 * <I>Special Notes:</I>
 *           Does not try to free memory if fed with a NULL pointer
 */
extern PVOID
CONTROL_Free_KMemory (
    PVOID  location,
    size_t size
)
{
    SEP_DRV_LOG_ALLOC_IN("Will free %p (%dB)...", location, size);

    if (location) {
        contigfree(location, size, M_SEP);
    }

    SEP_DRV_LOG_ALLOC_OUT("Freed %p (%dB).", location, size);
    return NULL;
}
