/*COPYRIGHT**
 *
 * Copyright (C) 2013 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.















**COPYRIGHT*/





#include "lwpmudrv_defines.h"
#include "lwpmudrv_types.h"
#include "inc/control.h"
#include "inc/utility.h"
#include "inc/sepdrv_p_state.h"

/*!
 * @fn     OS_STATUS SEPDRV_P_STATE_Read
 *
 * @brief  Reads the APERF and MPERF counters into the buffer provided for the purpose
 *
 * @param  buffer  - buffer to read the counts into
 *
 * @param  pcpu - pcpu struct that contains the previous APERF/MPERF values
 *
 * @return OS_SUCCESS if read succeeded, otherwise error
 *
 * @note
 */
extern OS_STATUS
SEPDRV_P_STATE_Read(S8 *buffer, CPU_STATE pcpu)
{
	U64 *samp      = (U64 *)buffer;
	U64  new_APERF = 0;
	U64  new_MPERF = 0;

	SEP_DRV_LOG_TRACE_IN("Buffer: %p, pcpu: %p.", buffer, pcpu);

	if ((samp == NULL) || (pcpu == NULL)) {
		SEP_DRV_LOG_ERROR_TRACE_OUT("OS_INVALID (!samp || !pcpu).");
		return OS_INVALID;
	}

	new_APERF = SYS_Read_MSR(DRV_APERF_MSR);
	new_MPERF = SYS_Read_MSR(DRV_MPERF_MSR);

	if (CPU_STATE_last_p_state_valid(pcpu)) {
		// there is a previous APERF/MPERF value
		if ((CPU_STATE_last_aperf(pcpu)) > new_APERF) {
			// a wrap-around has occurred.
			samp[1] = CPU_STATE_last_aperf(pcpu) - new_APERF;
		} else {
			samp[1] = new_APERF - CPU_STATE_last_aperf(pcpu);
		}

		if ((CPU_STATE_last_mperf(pcpu)) > new_MPERF) {
			// a wrap-around has occurred.
			samp[0] = CPU_STATE_last_mperf(pcpu) - new_MPERF;
		} else {
			samp[0] = new_MPERF - CPU_STATE_last_mperf(pcpu);
		}
	} else {
		// there is no previous valid APERF/MPERF values, thus no delta calculations
		(CPU_STATE_last_p_state_valid(pcpu)) = TRUE;
		samp[0]                              = 0;
		samp[1]                              = 0;
	}

	CPU_STATE_last_aperf(pcpu) = new_APERF;
	CPU_STATE_last_mperf(pcpu) = new_MPERF;

	SEP_DRV_LOG_TRACE_OUT("OS_SUCCESS.");
	return OS_SUCCESS;
}

