/* CLI utilities.

   Copyright (C) 2011-2025 Free Software Foundation, Inc.
   Copyright (C) 2019-2025 Intel Corporation

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

#ifndef CLI_CLI_UTILS_H
#define CLI_CLI_UTILS_H
#include <set>
#include <cstring>

#include "completer.h"

struct cmd_list_element;

/* Return values for get_number_overflow and get_number_trailer_overflow since
   bool is not enough for some cases.  See handling of thread ranges.  */
enum get_number_status
{
  NUMBER_OK = 0,
  NUMBER_ERROR,
  NUMBER_CONVERSION_ERROR
};

/* *PP is a string denoting a number.  Get the number.  Advance *PP
   after the string and any trailing whitespace.

   The string can either be a number, or "$" followed by the name of a
   convenience variable, or ("$" or "$$") followed by digits.

   TRAILER is a character which can be found after the number; most
   commonly this is `-'.  If you don't want a trailer, use \0.

   Returns TRUE on success (parsed value is written to PARSED_VALUE).
   If parsing failed, returns FALSE.

   Internally calls get_number_trailer_overflow ().  */

extern bool get_number_trailer (const char **pp, int *parsed_value,
				int trailer);

/* Backend for get_number_trailer, does overflow checking.

   Returns NUMBER_OK on success (parsed value is written to PARSED_VALUE).
   If parsing failed, returns NUMBER_ERROR or NUMBER_CONVERSION_ERROR.

   NUMBER_CONVERSION_ERROR is returned if strtol () conversion to int overflows.

   NUMBER_ERROR is returned for all other parsing errors.  */

extern get_number_status
get_number_trailer_overflow (const char **pp, int *parsed_value, int trailer);

/* Convenience.  Like get_number_trailer, but with no TRAILER.  */

extern bool get_number (const char **, int *);

/* Convenience.  Like get_number_trailer, but with no TRAILER and can
   return more detailed result codes.  Above get_number is just wrapper around
   this.  */

extern get_number_status get_number_overflow (const char **, int *);

/* Like the above, but takes a non-const "char **".  */
extern bool get_number (char **, int *);

/* Like the above, but takes a non-const "char **" and returns an extended
   result.  */
extern get_number_status get_number_overflow (char **, int *);

/* Like get_number_trailer, but works with ULONGEST, and throws on
   error instead of returning an error enum.  */
extern ULONGEST get_ulongest (const char **pp, int trailer = '\0');

/* Throws an error telling the user that ARGS starts with an option
   unrecognized by COMMAND.  */

extern void report_unrecognized_option_error (const char *command,
					      const char *args);


/* Builds the help string for a command documented by PREFIX,
   followed by the extract_info_print_args help for ENTITY_KIND.  If
   DOCUMENT_N_FLAG is true then help text describing the -n flag is also
   included.  */

const char *info_print_args_help (const char *prefix,
				  const char *entity_kind,
				  bool document_n_flag);

/* Parse a number or a range.
   A number will be of the form handled by get_number.
   A range will be of the form <number1> - <number2>, and
   will represent all the integers between number1 and number2,
   inclusive.  */

class number_or_range_parser
{
public:
  /* Default construction.  Must call init before calling
     get_next.  */
  number_or_range_parser () {}

  /* Calls init automatically.  */
  number_or_range_parser (const char *string);

  /* STRING is the string to be parsed.  END_TRAILER is optional and
     specifies an additional symbol which marks the valid end of
     a number.  */
  void init (const char *string, int end_trailer = 0);

  /* While processing a range, this fuction is called iteratively; At
     each call it will return the next value in the range.

     At the beginning of parsing a range, the char pointer
     STATE->m_cur_tok will be advanced past <number1> and left
     pointing at the '-' token.  Subsequent calls will not advance the
     pointer until the range is completed.  The call that completes
     the range will advance the pointer past <number2>.  */
  bool get_number (int *num);

  /* Like get_number but allows more subtle result codes.
     Currently this can return if a general error occurred or if a number
     was out of bounds in conversion to integer.  */
  enum get_number_status get_number_overflow (int *num);

  /* Setup internal state such that get_next() returns numbers in the
     START_VALUE to END_VALUE range.  END_PTR is where the string is
     advanced to when get_next() returns END_VALUE.  */
  void setup_range (int start_value, int end_value,
		    const char *end_ptr);

  /* Returns true if parsing has completed.  */
  bool finished () const;

  /* Return the string being parsed.  When parsing has finished, this
     points past the last parsed token.  */
  const char *cur_tok () const
  { return m_cur_tok; }

  /* True when parsing a set of numbers encapsulated into '[..]'.  */
  bool in_set () const
  { return m_in_set; };

  /* True when parsing a range.  */
  bool in_range () const
  { return m_in_range; }

  /* When parsing a range, the final value in the range.  */
  int end_value () const
  { return m_end_value; }

  /* When parsing a range, skip past the final token in the range.  */
  void skip_range ()
  {
    gdb_assert (m_in_range);
    m_cur_tok = m_end_ptr;
    m_in_range = false;
  }

  /* When parsing a set of numbers and/or ranges enclosed in square
     brackets (e.g., '[2 4-6 8]'), skip to the closing square bracket.  */
  void skip_set ()
  {
    gdb_assert (m_in_set);
    if (m_end_ptr != nullptr)
      m_cur_tok = m_end_ptr;
    else
      {
	const char* bracket = strchr (m_cur_tok, ']');
	if (bracket != nullptr)
	  m_cur_tok = bracket + 1;
      }

    m_in_set = false;
  }

  /* Setup the END_PTR, where the string is advanced to when get_next()
     returns END_VALUE.  */
  void set_end_ptr (const char*);

private:
  /* No need for these.  They are intentionally not defined anywhere.  */
  number_or_range_parser (const number_or_range_parser &);
  number_or_range_parser &operator= (const number_or_range_parser &);

  /* The string being parsed.  When parsing has finished, this points
     past the last parsed token.  */
  const char *m_cur_tok;

  /* Last value returned.  */
  int m_last_retval;

  /* When parsing a range, the final value in the range.  */
  int m_end_value;
  int m_end_trailer;

  /* When parsing a range or set, a pointer past the final token
     in the range/set.  */
  const char *m_end_ptr;

  /* True when parsing a range.  */
  bool m_in_range;

  /* True when parsing a set of numbers encapsulated into square brackets.  */
  bool m_in_set;
};

/* Accept a number and a string-form list of numbers such as is 
   accepted by get_number_or_range.  Return TRUE if the number is
   in the list.

   By definition, an empty list includes all numbers.  This is to 
   be interpreted as typing a command such as "delete break" with 
   no arguments.  */

extern int number_is_in_list (const char *list, int number);

/* Reverse S to the last non-whitespace character without skipping past
   START.  */

extern const char *remove_trailing_whitespace (const char *start,
					       const char *s);

/* Same, for non-const S.  */

static inline char *
remove_trailing_whitespace (const char *start, char *s)
{
  return (char *) remove_trailing_whitespace (start, (const char *) s);
}

/* Same as 'skip_to_space' if CHP does not contain square brackets.
   If CHP contains square brackets, skip to closing bracket.  */
extern const char *skip_to_next (const char *chp);

/* A helper function to extract an argument from *ARG.  An argument is
   delimited by whitespace.  The return value is empty if no argument
   was found.  */

extern std::string extract_arg (char **arg);

/* A const-correct version of the above.  */

extern std::string extract_arg (const char **arg);

/* A helper function that looks for an argument at the start of a
   string.  The argument must also either be at the end of the string,
   or be followed by whitespace.  Returns 1 if it finds the argument,
   0 otherwise.  If the argument is found, it updates *STR to point
   past the argument and past any whitespace following the
   argument.  */
extern int check_for_argument (const char **str, const char *arg, int arg_len);

/* Same as above, but ARG's length is computed.  */

static inline int
check_for_argument (const char **str, const char *arg)
{
  return check_for_argument (str, arg, strlen (arg));
}

/* Same, for non-const STR.  */

static inline int
check_for_argument (char **str, const char *arg, int arg_len)
{
  return check_for_argument (const_cast<const char **> (str),
			     arg, arg_len);
}

static inline int
check_for_argument (char **str, const char *arg)
{
  return check_for_argument (str, arg, strlen (arg));
}

/* qcs_flags struct groups the -q, -c, and -s flags parsed by "thread
   apply" and "frame apply" commands */

struct qcs_flags
{
  bool quiet = false;
  bool cont = false;
  bool silent = false;
};

/* Validate FLAGS.  Throws an error if both FLAGS->CONT and
   FLAGS->SILENT are true.  WHICH_COMMAND is included in the error
   message.  */
extern void validate_flags_qcs (const char *which_command, qcs_flags *flags);

/* Create a string of ranges out of MASK.  If the result contains more than
   one number, it is enclosed in square brackets.  If CURRENT is used and
   matches one of the bits set in MASK, it will be prepended by a '*' and not
   be part of any range.

   Example:

   For the mask '0' the result is "".
   For the mask '1' the result is "1".
   For the mask '0b1 1101 0111' the result is "[0-2 4 6-8]".
   For the mask '15' with current lane 0 the result is "[*0 1-3]".
   For the mask '15' with current lane 1 the result is "[0 *1 2-3]".
   For the mask '15' with current lane 2 the result is "[0-1 *2 3]".
   For the mask '15' with current lane 3 the result is "[0-2 *3]".  */

extern std::string make_ranges_from_mask (unsigned long mask,
					  int current = -1);

#endif /* CLI_CLI_UTILS_H */
