/*******************************************************************************
* Copyright (C) 2014 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

//@HEADER
// ***************************************************
//
// HPCG: High Performance Conjugate Gradient Benchmark
//
// Contact:
// Michael A. Heroux ( maherou@sandia.gov)
// Jack Dongarra     (dongarra@eecs.utk.edu)
// Piotr Luszczek    (luszczek@eecs.utk.edu)
//
// ***************************************************
//@HEADER

/*!
 @file UsmUtil.cpp

 HPCG routine
 */
#include "UsmUtil.hpp"

using namespace sycl;

void * sparse_malloc_host(size_t bytes, sycl::queue & main_queue) {
  void *x = nullptr;
  x = (void*) sycl::malloc_host(bytes, main_queue);
  if (bytes > 0 && x == nullptr)
      throw std::runtime_error("Allocation failed");
  return x;
}

void * sparse_malloc_shared(size_t bytes, sycl::queue & main_queue) {
  void *x = nullptr;
  x = (void*) sycl::malloc_shared(bytes, main_queue);
  if (bytes > 0 && x == nullptr)
      throw std::runtime_error("Allocation failed");
  return x;
}

void * sparse_malloc_device(size_t bytes, sycl::queue & main_queue) {
  void *x = nullptr;
  x = (void*) sycl::malloc_device(bytes, main_queue);
  if (bytes > 0 && x == nullptr)
      throw std::runtime_error("Allocation failed");
  return x;
}

/*!
  Copy input vector to output vector.

  @param[in] v Input vector
  @param[in] w Output vector
 */
sycl::event CopyVector(const Vector & v, Vector & w, sycl::queue & main_queue,
                       const std::vector<sycl::event> &deps) {
  local_int_t localLength = v.localLength;
  assert(w.localLength >= localLength);
  double * vv = v.values;
  double * wv = w.values;
  return main_queue.memcpy(wv, vv, localLength * sizeof(double), deps);
}

/*!
  Fill the input vector with zero values.

  @param[inout] v - On entrance v is initialized, on exit all its values are zero.
 */
sycl::event ZeroVector(const Vector & v, sycl::queue & main_queue,
                       const std::vector<sycl::event> & deps) {
  local_int_t localLength = v.localLength;
  double * vv = v.values;
  return main_queue.fill<double>(vv, 0.0, localLength, deps);
}
