/*******************************************************************************
* Copyright (C) 2021 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP API oneapi::mkl::sparse::gemm to perform general sparse matrix-matrix
*       multiplication on a SYCL device (CPU, GPU). This example uses
*       column-major layout for dense matrices X and Y, and CSR format for
*       sparse matrix A.
*
*       C = alpha * op(A) * op(X) + beta * Y
*
*       where op() is defined by one of
*           oneapi::mkl::transpose::{nontrans,trans,conjtrans}
*
*       The supported floating point data types for gemm are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported transpose operation op() for sparse matrix A are:
*           oneapi::mkl::transpose::nontrans
*           oneapi::mkl::transpose::trans
*           oneapi::mkl::transpose::conjtrans
*
*       The supported transpose operation op() for dense matrix X are:
*           oneapi::mkl::transpose::nontrans
*
*       The supported matrix formats for gemm are:
*           CSR
*           COO (currently only on CPU device)
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"
#include "./include/reference_impls.hpp"

//
// Main example for Sparse Matrix-Dense Matrix Multiply consisting of
// initialization of sparse A matrix, dense X and Y matrices as well as
// scalars alpha and beta.  Then the product
//
// Y = alpha * op(A) * op(X) + beta * Y
//
// is performed and finally the results are post processed.
//
template <typename dataType, typename intType>
int run_sparse_blas_example(sycl::queue &q)
{
    bool good = true;

    //
    // handle for sparse matrix
    //
    oneapi::mkl::sparse::matrix_handle_t csrA = nullptr;

    try {

        // Initialize data for Sparse Matrix-Vector Multiply
        oneapi::mkl::transpose opA = oneapi::mkl::transpose::trans;
        if constexpr (is_complex<dataType>()) { opA = oneapi::mkl::transpose::nontrans; }

        oneapi::mkl::transpose opX              = oneapi::mkl::transpose::nontrans;
        oneapi::mkl::layout dense_matrix_layout = oneapi::mkl::layout::col_major;
        oneapi::mkl::index_base index_base_val  = oneapi::mkl::index_base::zero;

        intType int_index = (index_base_val == oneapi::mkl::index_base::zero ? 0 : 1);

        // Matrix data size
        const std::int64_t nrows_a = 64;
        const std::int64_t ncols_a = 128;
        const std::int64_t columns = 256;
        const std::int64_t ldx     = (opA == oneapi::mkl::transpose::nontrans) ? ncols_a : nrows_a;
        const std::int64_t ldy     = (opA == oneapi::mkl::transpose::nontrans) ? nrows_a : ncols_a;

        double density_val = 0.05;

        // Input matrix in CSR format
        std::vector<intType, mkl_allocator<intType, 64>> ia;
        std::vector<intType, mkl_allocator<intType, 64>> ja;
        std::vector<dataType, mkl_allocator<dataType, 64>> a;

        generate_random_sparse_matrix<dataType, intType>(nrows_a, ncols_a, density_val, ia, ja, a, int_index);
        const std::int64_t nnz_a = ia[nrows_a] - int_index;

        // Matrices X and Y
        std::vector<dataType, mkl_allocator<dataType, 64>> dnsX;
        std::vector<dataType, mkl_allocator<dataType, 64>> dnsY;
        std::vector<dataType, mkl_allocator<dataType, 64>> dnsY_ref;

        const std::int64_t nrows_opa = (oneapi::mkl::transpose::nontrans == opA) ? nrows_a : ncols_a;
        const std::int64_t ncols_opa = (oneapi::mkl::transpose::nontrans == opA) ? ncols_a : nrows_a;

        const std::int64_t nrows_x = ncols_opa;
        const std::int64_t ncols_x = columns;
        const std::int64_t nrows_y = nrows_opa;
        const std::int64_t ncols_y = columns;

        rand_matrix<std::vector<dataType, mkl_allocator<dataType, 64>>>(dnsX, dense_matrix_layout,
                                                            nrows_x, ncols_x, ldx);
        dnsX.resize(ldx * ncols_x);
        dnsY.resize(ldy * ncols_y);
        dnsY_ref.resize(ldy * ncols_y);

        // Init matrices Y and Y_ref
        for (int i = 0; i < dnsY.size(); i++) {
            dnsY[i]     = set_fp_value(dataType(0.0), dataType(0.0));
            dnsY_ref[i] = set_fp_value(dataType(0.0), dataType(0.0));
        }

        // Set scalar dataType values
        dataType alpha = set_fp_value(dataType(2.0), dataType(0.0));
        dataType beta  = set_fp_value(dataType(1.0), dataType(0.0));

        //
        // Execute Matrix Multiply
        //

        std::cout << "\n\t\tsparse::gemm parameters:\n";
        std::cout << "\t\t\tdense_matrix_layout = " << dense_matrix_layout << std::endl;
        std::cout << "\t\t\topA                 = " << opA << std::endl;
        std::cout << "\t\t\topX                 = " << opX << std::endl;
        std::cout << "\t\t\tnrows               = " << nrows_a << std::endl;
        std::cout << "\t\t\tncols               = " << ncols_a << std::endl;
        std::cout << "\t\t\tnnz                 = " << nnz_a << std::endl;
        std::cout << "\t\t\tcolumns             = " << columns << std::endl;
        std::cout << "\t\t\tldx                 = " << ldx << std::endl;
        std::cout << "\t\t\tldy                 = " << ldy << std::endl;
        std::cout << "\t\t\talpha               = " << alpha << std::endl;
        std::cout << "\t\t\tbeta                = " << beta << std::endl;

        sycl::buffer<intType, 1> ia_buf(ia.data(), ia.data() + nrows_a + 1);
        sycl::buffer<intType, 1> ja_buf(ja.data(), ja.data() + nnz_a);
        sycl::buffer<dataType, 1> a_buf(a.data(), a.data() + nnz_a);
        sycl::buffer<dataType, 1> dnsX_buf(dnsX.data(), dnsX.data() + dnsX.size());
        sycl::buffer<dataType, 1> dnsY_buf(dnsY.data(), dnsY.data() + dnsY.size());

        oneapi::mkl::sparse::init_matrix_handle(&csrA);

        oneapi::mkl::sparse::set_csr_data(q, csrA, nrows_a, ncols_a, nnz_a, index_base_val,
                                          ia_buf, ja_buf, a_buf);

        // add oneapi::mkl::sparse::gemm to execution queue
        oneapi::mkl::sparse::gemm(q, dense_matrix_layout, opA, opX, alpha,
                                  csrA, dnsX_buf, columns, ldx, beta, dnsY_buf, ldy);

        oneapi::mkl::sparse::release_matrix_handle(q, &csrA);

        q.wait();

        //
        // Post Processing
        //

        // Compute reference gemm result.
        // NOTE: Now we support only opX == nontrans case, so we don't pass it as argument.
        prepare_reference_gemm_data(ia.data(), ja.data(), a.data(), nrows_a, ncols_a, nnz_a, int_index,
                opA, alpha, beta, dense_matrix_layout, dnsX.data(), ncols_x, ldx, dnsY_ref.data(), ldy);

        dataType diff  = set_fp_value(dataType(0.0), dataType(0.0));
        dataType diff2 = set_fp_value(dataType(0.0), dataType(0.0));
        auto res = dnsY_buf.get_host_access(sycl::read_only);
        for (intType j = 0; j < ncols_y; j++) {
            intType flops_for_val = 2 * (ceil_div(nnz_a, nrows_y) + 2);
            for (intType i = 0; i < nrows_y; i++) {
                if (opA == oneapi::mkl::transpose::nontrans) {
                    flops_for_val = 2*(ia[i+1] - ia[i] + 2);
                }

                intType index = i + j * ldy;
                if (!check_result(res[index], dnsY_ref[index], flops_for_val, index)) {
                    good = false;
                }
                diff += (dnsY_ref[index] - res[index]) * (dnsY_ref[index] - res[index]);
                diff2 += dnsY_ref[index] * dnsY_ref[index];
            }
        }

        std::cout << "\n\t\t sparse::gemm residual:\n"
                  << "\t\t\t" << diff / diff2 << "\n\tFinished" << std::endl;

        q.wait_and_throw();

    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;
        good = false;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;
        good = false;
    }

    q.wait();

    // backup cleaning of matrix handle and others for if exceptions happened
    oneapi::mkl::sparse::release_matrix_handle(q, &csrA);

    return good ? 0 : 1;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# Sparse Matrix-Dense Matrix Multiply Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Y = alpha * op(A) * op(X) + beta * Y" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A is a sparse matrix in CSR format, X and Y are "
                 "dense matrices"
              << std::endl;
    std::cout << "# and alpha, beta are floating point type precision scalars." << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::gemm" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type,
//  run_sparse_blas_example is run with all supported data types,
//  if any fail, we move on to the next device.
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {
        try {
            sycl::device my_dev;
            bool my_dev_is_found = false;
            get_sycl_device(my_dev, my_dev_is_found, *it);

            if (my_dev_is_found) {
                std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

                // Catch asynchronous exceptions
                auto exception_handler = [](sycl::exception_list exceptions) {
                    for (std::exception_ptr const &e : exceptions) {
                        try {
                            std::rethrow_exception(e);
                        }
                        catch (sycl::exception const &e) {
                            std::cout << "Caught asynchronous SYCL exception: \n"
                                << e.what() << std::endl;
                        }
                    }
                };

                sycl::queue q(my_dev, exception_handler);

                std::cout << "\tRunning with single precision real data type:" << std::endl;
                status |= run_sparse_blas_example<float, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision real data type:" << std::endl;
                    status |= run_sparse_blas_example<double, std::int32_t>(q);
                }

                std::cout << "\tRunning with single precision complex data type:" << std::endl;
                status |= run_sparse_blas_example<std::complex<float>, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision complex data type:" << std::endl;
                    status |= run_sparse_blas_example<std::complex<double>, std::int32_t>(q);
                }

            }
            else {
#ifdef FAIL_ON_MISSING_DEVICES
                std::cout << "No " << sycl_device_names[*it]
                    << " devices found; Fail on missing devices "
                    "is enabled.\n";
                return 1;
#else
                std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                    << sycl_device_names[*it] << " tests.\n";
#endif
            }
        }
        catch (sycl::exception const &e) {
            std::cout << "\t\tCaught SYCL exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }
        catch (std::exception const &e) {
            std::cout << "\t\tCaught std exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }


    } // for loop over devices

    mkl_free_buffers();
    return status;
}
